/**
 * Respond Popup Script
 * Toggle + manual refresh + local accounts
 */

document.addEventListener("DOMContentLoaded", async () => {
  const enableToggle = document.getElementById("enableToggle");
  const refreshBtn = document.getElementById("refreshBtn");
  const accountInput = document.getElementById("accountInput");
  const addBtn = document.getElementById("addBtn");
  const localAccountsDiv = document.getElementById("localAccounts");

  // Load saved state
  const result = await chrome.storage.local.get(["enabled", "localAccounts"]);
  enableToggle.checked = result.enabled !== false;

  // Render local accounts
  let localAccounts = result.localAccounts || [];
  renderLocalAccounts();

  // Toggle handler
  enableToggle.addEventListener("change", async () => {
    const enabled = enableToggle.checked;
    await chrome.storage.local.set({ enabled });
    sendToActiveTab({ action: "toggle", enabled });
  });

  // Refresh button
  refreshBtn.addEventListener("click", async () => {
    refreshBtn.textContent = "↻ Refreshing...";
    refreshBtn.disabled = true;

    await sendToActiveTab({ action: "refresh" });

    setTimeout(() => {
      refreshBtn.textContent = "✓ Updated!";
      setTimeout(() => {
        refreshBtn.textContent = "↻ Refresh Now";
        refreshBtn.disabled = false;
      }, 2000);
    }, 1000);
  });

  // Add account
  addBtn.addEventListener("click", addAccount);
  accountInput.addEventListener("keypress", (e) => {
    if (e.key === "Enter") addAccount();
  });

  async function addAccount() {
    let username = accountInput.value.trim().toLowerCase().replace("@", "");
    if (!username) return;

    if (!localAccounts.includes(username)) {
      localAccounts.push(username);
      await chrome.storage.local.set({ localAccounts });
      renderLocalAccounts();
      sendToActiveTab({
        action: "updateLocalAccounts",
        accounts: localAccounts,
      });
    }
    accountInput.value = "";
  }

  async function removeAccount(username) {
    localAccounts = localAccounts.filter((a) => a !== username);
    await chrome.storage.local.set({ localAccounts });
    renderLocalAccounts();
    sendToActiveTab({ action: "updateLocalAccounts", accounts: localAccounts });
  }

  function renderLocalAccounts() {
    localAccountsDiv.innerHTML = localAccounts
      .map(
        (account) => `
      <span class="local-tag">
        @${account}
        <button data-account="${account}" title="Remove">×</button>
      </span>
    `,
      )
      .join("");

    // Add remove handlers
    localAccountsDiv.querySelectorAll("button").forEach((btn) => {
      btn.addEventListener("click", () => removeAccount(btn.dataset.account));
    });
  }
});

async function sendToActiveTab(message) {
  try {
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });
    if (
      tab?.id &&
      (tab.url?.includes("twitter.com") || tab.url?.includes("x.com"))
    ) {
      await chrome.tabs.sendMessage(tab.id, message);
    }
  } catch (e) {
    // Tab might not have content script
  }
}
