/**
 * PunchThrough Content Script
 *
 * Fetches messages and accounts from persia.free for remote updates.
 * LOCKED DOWN: Only fetches from persia.free, cannot be customized.
 */

// Remote config URL
const CONFIG_URL = "https://persia.free/extension/config.json";

// Fallback defaults (used if fetch fails)
const DEFAULT_MESSAGES = [
  "The people of Iran are being crushed by the Ayatollah regime. We ask the free world to stand with us and make sure their abuses are seen and condemned. Read our message here https://persia.free",
  "Iranian youth are being killed in the streets, they are unarmed and asking for dignity. The world should listen to the people of Iran, not the Ayatollah regime's propaganda. Help shine a light on what is happening and read our statement https://persia.free",
];

const DEFAULT_ACCOUNTS = ["araghchi", "presstv", "irna_en", "khamenei_ir"];

// Runtime state
let messages = [...DEFAULT_MESSAGES];
let remoteAccounts = new Set(DEFAULT_ACCOUNTS);
let localAccounts = new Set();
let messageIndex = 0;
let isEnabled = true;
let lastFetch = 0;

// Combined accounts (remote + local)
function getAllAccounts() {
  return new Set([...remoteAccounts, ...localAccounts]);
}
// Refresh once per 24 hours + random jitter (±2 hours) to spread server load
const FETCH_INTERVAL = 24 * 60 * 60 * 1000; // 24 hours
const JITTER = Math.random() * 4 * 60 * 60 * 1000 - 2 * 60 * 60 * 1000; // ±2 hours

/**
 * Fetch config from persia.free
 */
async function fetchRemoteConfig() {
  try {
    const response = await fetch(CONFIG_URL + "?t=" + Date.now(), {
      cache: "no-store",
    });

    if (!response.ok) throw new Error("Fetch failed");

    const config = await response.json();

    // Update messages
    if (
      config.messages &&
      Array.isArray(config.messages) &&
      config.messages.length > 0
    ) {
      messages = config.messages;
      console.log("🕊️ Messages updated:", messages.length);
    }

    // Update accounts
    if (
      config.accounts &&
      Array.isArray(config.accounts) &&
      config.accounts.length > 0
    ) {
      remoteAccounts = new Set(
        config.accounts.map((a) => a.toLowerCase().replace("@", "")),
      );
      console.log("🕊️ Remote accounts updated:", remoteAccounts.size);
    }

    // Save to local storage as cache
    await chrome.storage.local.set({
      cachedMessages: messages,
      cachedAccounts: [...remoteAccounts],
      lastFetch: Date.now(),
    });

    lastFetch = Date.now();

    // Rescan page with new config
    rescanAll();
  } catch (error) {
    console.log("🕊️ Using cached/default config:", error.message);
  }
}

/**
 * Initialize from cache, then fetch fresh
 */
async function initConfig() {
  try {
    const result = await chrome.storage.local.get([
      "enabled",
      "messageIndex",
      "cachedMessages",
      "cachedAccounts",
      "localAccounts",
      "lastFetch",
    ]);

    isEnabled = result.enabled !== false;
    messageIndex = result.messageIndex || 0;
    lastFetch = result.lastFetch || 0;

    // Load from cache first (instant)
    if (result.cachedMessages && result.cachedMessages.length > 0) {
      messages = result.cachedMessages;
    }
    if (result.cachedAccounts && result.cachedAccounts.length > 0) {
      remoteAccounts = new Set(result.cachedAccounts);
    }
    if (result.localAccounts && result.localAccounts.length > 0) {
      localAccounts = new Set(result.localAccounts);
    }

    console.log(
      "🕊️ Respond loaded:",
      remoteAccounts.size,
      "remote +",
      localAccounts.size,
      "local accounts,",
      messages.length,
      "messages",
    );
  } catch (e) {
    console.log("Respond: Using defaults");
  }

  // Fetch fresh config if stale (24h + jitter)
  if (Date.now() - lastFetch > FETCH_INTERVAL + JITTER) {
    await fetchRemoteConfig();
  }
}

/**
 * Get next message (rotating) and save index
 */
function getNextMessage() {
  const msg = messages[messageIndex];
  messageIndex = (messageIndex + 1) % messages.length;
  try {
    chrome.storage.local.set({ messageIndex });
  } catch (e) {
    // Extension context invalidated - page needs refresh
  }
  return msg;
}

/**
 * Check if username is a regime account
 */
function isRegimeAccount(username) {
  if (!username) return false;
  const allAccounts = getAllAccounts();
  return allAccounts.has(username.toLowerCase().replace("@", ""));
}

/**
 * Create the PunchThrough button
 */
function createPunchButton(tweetElement) {
  const button = document.createElement("button");
  button.className = "punch-through-btn";
  button.innerHTML = '<span class="icon-circle">🕊️</span> Respond';
  button.title = "Respond with a message of peace";

  button.addEventListener("click", async (e) => {
    e.preventDefault();
    e.stopPropagation();

    const replyButton = tweetElement.querySelector('[data-testid="reply"]');
    if (replyButton) {
      replyButton.click();

      setTimeout(() => {
        const composeBox = document.querySelector(
          '[data-testid="tweetTextarea_0"]',
        );
        if (composeBox) {
          const message = getNextMessage();
          composeBox.focus();
          document.execCommand("insertText", false, message);

          button.innerHTML = '<span class="icon-circle">✅</span> Ready!';
          button.classList.add("punch-success");

          setTimeout(() => {
            button.innerHTML = '<span class="icon-circle">🕊️</span> Respond';
            button.classList.remove("punch-success");
          }, 3000);
        }
      }, 500);
    }
  });

  return button;
}

/**
 * Process a single tweet
 */
function processTweet(tweetElement) {
  if (tweetElement.dataset.punchProcessed) return;
  tweetElement.dataset.punchProcessed = "true";

  const userLinks = tweetElement.querySelectorAll('a[href^="/"]');
  let username = null;

  for (const link of userLinks) {
    const href = link.getAttribute("href");
    if (href && href.match(/^\/[a-zA-Z0-9_]+$/) && !href.includes("/status")) {
      username = href.substring(1);
      break;
    }
  }

  // Add Respond button to ALL tweets
  const actionBar = tweetElement.querySelector('[role="group"]');
  if (actionBar && !actionBar.querySelector(".punch-through-btn")) {
    const button = createPunchButton(tweetElement);
    actionBar.appendChild(button);
  }

  // Only add badge and highlight for regime accounts
  if (username && isRegimeAccount(username)) {
    tweetElement.classList.add("regime-tweet");

    if (!tweetElement.querySelector(".regime-badge")) {
      const badge = document.createElement("span");
      badge.className = "regime-badge";
      badge.innerHTML = "⚠ Regime Propaganda";
      badge.title = "This account spreads Ayatollah regime propaganda";

      const header = tweetElement.querySelector('[data-testid="User-Name"]');
      if (header) {
        header.appendChild(badge);
      }
    }
  }
}

/**
 * Scan page for tweets
 */
function scanForTweets() {
  if (!isEnabled) return;
  const tweets = document.querySelectorAll('[data-testid="tweet"]');
  tweets.forEach(processTweet);
}

/**
 * Force rescan
 */
function rescanAll() {
  document.querySelectorAll('[data-testid="tweet"]').forEach((t) => {
    delete t.dataset.punchProcessed;
    t.querySelectorAll(".regime-badge, .punch-through-btn").forEach((el) =>
      el.remove(),
    );
    t.classList.remove("regime-tweet");
  });
  scanForTweets();
}

/**
 * Initialize observer for dynamic content
 */
function initObserver() {
  const observer = new MutationObserver((mutations) => {
    let shouldScan = false;
    for (const mutation of mutations) {
      if (mutation.addedNodes.length > 0) {
        shouldScan = true;
        break;
      }
    }
    if (shouldScan) {
      setTimeout(scanForTweets, 100);
    }
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true,
  });
}

/**
 * Listen for messages from popup
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "toggle") {
    isEnabled = message.enabled;
    if (isEnabled) scanForTweets();
    sendResponse({ success: true });
  }

  if (message.action === "refresh") {
    fetchRemoteConfig();
    sendResponse({ success: true });
  }

  if (message.action === "updateLocalAccounts") {
    localAccounts = new Set(message.accounts || []);
    console.log("🕊️ Local accounts updated:", localAccounts.size);
    rescanAll();
    sendResponse({ success: true });
  }

  return true;
});

/**
 * Initialize
 */
async function init() {
  await initConfig();
  console.log("🕊️ Respond activated");
  scanForTweets();
  initObserver();

  // Periodic refresh
  setInterval(
    () => {
      if (Date.now() - lastFetch > FETCH_INTERVAL) {
        fetchRemoteConfig();
      }
    },
    5 * 60 * 1000,
  ); // Check every 5 minutes
}

if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", init);
} else {
  init();
}
